/*
 * c64dtvblitter.c - C64DTV blitter and DMA controller
 *
 * Written by
 *  M.Kiesel <mayne@users.sourceforge.net>
 *  H.Nuotio <hannu.nuotio@pp.inet.fi>
 *  Daniel Kahlin <daniel@kahlin.net>
 * Based on code by
 *  Marco van den Heuvel <blackystardust68@yahoo.com>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include "vice.h"

#include <string.h>

#include "c64mem.h"
#include "c64dtvmem.h"
#include "c64dtvflash.h"
#include "c64dtvblitter.h"
#include "c64dtvdma.h"
#include "vicii-mem.h"
#include "cmdline.h"
#include "lib.h"
#include "log.h"
#include "util.h"
#include "resources.h"
#include "maincpu.h"
#include "interrupt.h"
#include "alarm.h"

#ifdef HAS_TRANSLATION
#include "translate.h"
#endif

static log_t c64dtvblitter_log = LOG_ERR;

/* I/O of the blitter engine ($D3XX) */
BYTE c64dtvmem_blitter[0x20];

int blit_sourceA_off;
int blit_sourceB_off;
int blit_dest_off;
int blitter_busy;
int blitter_irq;
int blitter_on_irq;
int blitter_log_enabled = 0;

/* these should go into the snapshot when cycle exact operation happens */
static BYTE srca_data[4];
static int srca_data_offs;
static int srca_fetched;
static BYTE srcb_data[4];
static int srcb_data_offs;
static BYTE sourceA, sourceB;
static int blitter_count;
static enum { BLITTER_IDLE, BLITTER_READ_A, BLITTER_READ_B, BLITTER_WRITE } blitter_state; 
static int sourceA_line_off;
static int sourceB_line_off;
static int dest_line_off;
static BYTE lastA;


/* resource stuff */
static int dtvrevision;
static int have_blitter_bug;


#define GET_REG24(a) ((c64dtvmem_blitter[a+2]<<16) | (c64dtvmem_blitter[a+1]<<8) | c64dtvmem_blitter[a])
#define GET_REG16(a) ((c64dtvmem_blitter[a+1]<<8) | c64dtvmem_blitter[a])
#define GET_REG8(a) (c64dtvmem_blitter[a])


/* ------------------------------------------------------------------------- */

void c64dtvblitter_init(void)
{
  if(c64dtvblitter_log == LOG_ERR)
    c64dtvblitter_log = log_open("C64DTVBLITTER");

  /* init Blitter IRQ */
  c64dtv_blitter_irq_init();
}


void c64dtvblitter_shutdown(void)
{
}

void c64dtvblitter_reset(void)
{
  int i;
  if(blitter_log_enabled) log_message(c64dtvblitter_log, "reset");
  /* TODO move register file initialization somewhere else? */
  for (i=0;i<0x20;++i) c64dtvmem_blitter[i] = 0;

  blit_sourceA_off = 0;
  blit_sourceB_off = 0;
  blit_dest_off = 0;
  blitter_busy = 0;
  blitter_irq = 0;
  blitter_on_irq = 0;

  blitter_count = 0;
  blitter_state = BLITTER_IDLE;
  srca_data_offs = -1;
  srcb_data_offs = -1;
  srca_fetched = 0;
  sourceA = 0;
  lastA = 0;
  sourceB = 0;
  sourceA_line_off = 0;
  sourceB_line_off = 0;
  dest_line_off = 0;
}

/* ------------------------------------------------------------------------- */
/* blitter transfer state machine */

static int do_blitter_read_a(void)
{
    int was_read = 0;
    int offs  = (blit_sourceA_off >> 4) & 0x1ffffc;
    int loffs = (blit_sourceA_off >> 4) & 0x000003;

    srca_fetched = 0;
    if (offs != srca_data_offs) {
        memcpy(srca_data, &mem_ram[offs], 4);
        srca_data_offs = offs;
	srca_fetched = 1;
	was_read = 1;
    }
    sourceA = srca_data[loffs];
    return was_read;
}

static int do_blitter_read_b(void)
{
    int force_sourceB_zero = GET_REG8(0x1b) & 0x01;

    int was_read = 0;
    int offs  = (blit_sourceB_off >> 4) & 0x1ffffc;
    int loffs = (blit_sourceB_off >> 4) & 0x000003;

    if (force_sourceB_zero) {
        sourceB = 0;
	return 0;
    }

    if (offs != srcb_data_offs) {
        memcpy(srcb_data, &mem_ram[offs], 4);
        srcb_data_offs = offs;
	was_read = 1;
    }
    sourceB = srcb_data[loffs];
    return was_read;
}


static int do_blitter_write(void)
{
    int sourceA_right_shift = GET_REG8(0x1e) & 0x07;
    int mintermALU = (GET_REG8(0x1e) >> 3) & 0x07;
    int write_if_sourceA_zero = GET_REG8(0x1b) & 0x02;
    int write_if_sourceA_nonzero = GET_REG8(0x1b) & 0x04;

    int was_write = 0;
    int offs  = (blit_dest_off >> 4) & 0x1fffff;


    if(!(write_if_sourceA_zero || write_if_sourceA_nonzero)) {
        write_if_sourceA_zero = write_if_sourceA_nonzero = 1;
    }
	
    if ( (write_if_sourceA_zero    && sourceA == 0) ||
	 (write_if_sourceA_nonzero && sourceA != 0) ||
         (have_blitter_bug && srca_fetched) ) {
        BYTE dest;
        BYTE lastA_tmp = sourceA;
	sourceA >>= sourceA_right_shift;
	sourceA |= lastA << (8 - sourceA_right_shift);
	lastA = lastA_tmp;

	dest = 0;
	switch(mintermALU) {
	case 0: dest = sourceA & sourceB; break;
	case 1: dest = ~(sourceA & sourceB); break;
	case 2: dest = ~(sourceA | sourceB); break;
	case 3: dest = sourceA | sourceB; break;
	case 4: dest = sourceA ^ sourceB; break;
	case 5: dest = ~(sourceA ^ sourceB); break;
	case 6: dest = sourceA + sourceB; break;
	case 7: dest = sourceA - sourceB; break;
	default:
	    break;
	}
	mem_ram[offs] = dest;
	was_write = 1;
    }
    if(blitter_log_enabled) log_message(c64dtvblitter_log, "Blitter: %s %x.%x/%x.%x to %x.%x, %d to go, minterm %d", was_write ? "transferred" : "skipped", blit_sourceA_off >> 4, blit_sourceA_off & 15, blit_sourceB_off >> 4, blit_sourceB_off & 15, blit_dest_off >> 4, blit_dest_off & 15, blitter_count - 1, mintermALU);
    return was_write;
}

static void update_counters(void)
{
    int sourceA_modulo = GET_REG16(0x03);
    int sourceA_line_length = GET_REG16(0x05);
    int sourceA_step = GET_REG8(0x07);
    int sourceB_modulo = GET_REG16(0x0b);
    int sourceB_line_length = GET_REG16(0x0d);
    int sourceB_step = GET_REG8(0x0f);
    int dest_modulo = GET_REG16(0x13);
    int dest_line_length = GET_REG16(0x15);
    int dest_step = GET_REG8(0x17);
    int sourceA_direction = (GET_REG8(0x1a)&0x02) ? +1 : -1;
    int sourceB_direction = (GET_REG8(0x1a)&0x04) ? +1 : -1;
    int dest_direction = (GET_REG8(0x1a)&0x08) ? +1 : -1;

    if(sourceA_line_off >= sourceA_line_length) {
        lastA = 0;
	sourceA_line_off = 0;
	blit_sourceA_off = ((blit_sourceA_off >> 4) + sourceA_modulo * sourceA_direction) << 4;
    } else {
        sourceA_line_off++;
	blit_sourceA_off += sourceA_step * sourceA_direction;
    }
    if(sourceB_line_off >= sourceB_line_length) {
        sourceB_line_off = 0;
	blit_sourceB_off = ((blit_sourceB_off >> 4) + sourceB_modulo * sourceB_direction) << 4;
    } else {
        sourceB_line_off++;
	blit_sourceB_off += sourceB_step * sourceB_direction;
    }
    if(dest_line_off >= dest_line_length) {
        dest_line_off = 0;
	blit_dest_off = ((blit_dest_off >> 4) + dest_modulo * dest_direction) << 4;
    } else {
        dest_line_off++;
	blit_dest_off += dest_step * dest_direction;
    }
}

/* 32 MHz processing clock */
#define SUBCYCLES 32
static void perform_blitter_cycle(void)
{
    int subcycle = 0;
    while (subcycle < SUBCYCLES) {
        switch (blitter_state) {
	case BLITTER_IDLE:
	    subcycle += SUBCYCLES;
	    break;
	case BLITTER_READ_A:
	    if (blitter_count == 0) {
	        blitter_state = BLITTER_IDLE;
		break;
	    }

	    if ( do_blitter_read_a() )
	        subcycle += SUBCYCLES;
	    blitter_state=BLITTER_READ_B;
	    break;
	case BLITTER_READ_B:
	    if ( do_blitter_read_b() )
	        subcycle += SUBCYCLES;
	    blitter_state=BLITTER_WRITE;
	    break;
	case BLITTER_WRITE:
	    if ( do_blitter_write() )
	        subcycle += SUBCYCLES;
	    else
	        subcycle += 1;

	    update_counters();
	    blitter_count--;

	    if (blitter_count==0)
	        blitter_state=BLITTER_IDLE;
	    else
	        blitter_state=BLITTER_READ_A;
	    break;
	default:
            log_message(c64dtvblitter_log, "invalid state in perform_blitter_cycle()");
            blitter_state=BLITTER_IDLE;
	    break;
	}
    }
}


/* ------------------------------------------------------------------------- */

/* These are the $D3xx Blitter register engine handlers */

/* Blitter IRQ code */

static unsigned int c64dtv_blitter_int_num;
struct alarm_s *c64dtv_blitter_irq_alarm;

void c64dtv_blitter_irq_alarm_handler(CLOCK offset, void *data)
{
    if(blitter_log_enabled) log_message(c64dtvblitter_log, "IRQ/Done");
    if(blitter_irq) {
        maincpu_set_irq(c64dtv_blitter_int_num, 1);
	blitter_busy = 2;
    }
    alarm_unset(c64dtv_blitter_irq_alarm);
    blitter_busy &= 0xfe;
    /* Scheduled DMA */
    if (dma_on_irq & 0x20) {
        dma_busy = 0;
        dma_on_irq = 0;
        c64dtv_dma_store(0x1f, (c64dtvmem_dma[0x1f] & 0x8f) | 1);
    }
}

void c64dtv_blitter_irq_init(void)
{
    c64dtv_blitter_int_num = interrupt_cpu_status_int_new(maincpu_int_status, "C64DTVBLITTER");

    c64dtv_blitter_irq_alarm = alarm_new(maincpu_alarm_context, "C64DTVBLITTERIrq",
                                         c64dtv_blitter_irq_alarm_handler, NULL);
}


void c64dtv_blitter_store(WORD addr, BYTE value)
{
  int blitter_triggered;
  int blitter_time;

  /* Store first, then check whether DMA access has been requested,
     perform if necessary. */
  c64dtvmem_blitter[addr] = value;

  /* Blitter code */
  blitter_triggered = GET_REG8(0x1a)&0x01;
  blitter_on_irq = GET_REG8(0x1a)&0x70;
  
  /* Clear Blitter IRQ */
  if((GET_REG8(0x1f)&0x01) && (blitter_busy==2)){
     if(blitter_log_enabled) log_message(c64dtvblitter_log, "Clear IRQ (%i)", blitter_busy);
     blitter_busy &= 0xfd;
     maincpu_set_irq(c64dtv_blitter_int_num, 0);
     blitter_irq = 0;
     /* reset clear IRQ strobe bit */
     c64dtvmem_blitter[0x1f] &= 0xfe;
  }
  
  if(blitter_on_irq && (blitter_busy==0)) {
      blitter_busy = 1;
      if(blitter_log_enabled) log_message(c64dtvblitter_log, "Scheduled Blitter (%02x)", blitter_on_irq);
      return;
  }
  
  if(blitter_triggered && (blitter_busy==0))
  {
    int sourceA_continue = GET_REG8(0x1f) & 0x02;
    int sourceB_continue = GET_REG8(0x1f) & 0x04;
    int dest_continue = GET_REG8(0x1f) & 0x08;

    /* last four bits of offsets are fractional */
    if(!sourceA_continue) {
        blit_sourceA_off = GET_REG24(0x00) & 0x3fffff;
        blit_sourceA_off <<= 4;
    }
    if(!sourceB_continue) {
        blit_sourceB_off = GET_REG24(0x08) & 0x3fffff;
        blit_sourceB_off <<= 4;
    }
    if(!dest_continue) {
        blit_dest_off = GET_REG24(0x10) & 0x3fffff;
        blit_dest_off <<= 4;
    }

    if(blitter_log_enabled && (sourceA_continue || sourceB_continue || dest_continue))
    {
        log_message(c64dtvblitter_log, "sourceA cont %s, sourceB cont %s, dest cont %s", sourceA_continue ? "on" : "off", sourceB_continue ? "on" : "off", dest_continue ? "on" : "off");
    }

    /* total number of bytes to transfer */
    blitter_count = GET_REG16(0x18);

    blitter_time = 0;

    /* initialize state variables */
    sourceA_line_off = 0;
    sourceB_line_off = 0;
    dest_line_off = 0;
    lastA = 0;
    srca_data_offs = -1;
    srcb_data_offs = -1;

    blitter_state = BLITTER_READ_A;
    do {
        perform_blitter_cycle();
	blitter_time++;
    } while (blitter_state != BLITTER_IDLE);

    if (GET_REG8(0x1a) & 0x80) {
	blitter_irq = 1;
    } else blitter_irq = 0;

    blitter_busy = 1;
    alarm_set(c64dtv_blitter_irq_alarm, maincpu_clk+blitter_time);

    /* reset blitter start reg */
    c64dtvmem_blitter[0x1a] = 0;
  }
}

/* ------------------------------------------------------------------------- */

/* These are called on read/writes to $D3xx */

BYTE REGPARM1 c64dtv_dmablit_read(WORD addr)
{
  if (!vicii_extended_regs())
    return vicii_read(addr);

  if((addr&0xff)==0x1f)
    return dma_busy;
  if((addr&0xff)==0x3f)
    return blitter_busy;
  /* the default return value is 0x00 too but I have seen some strangeness
     here.  I've seen something that looks like DMAed data. - tlr */
  return 0x00;
}


void REGPARM2 c64dtv_dmablit_store(WORD addr, BYTE value)
{
  if (!vicii_extended_regs()) {
      vicii_store(addr, value);
      return;
  }

  addr &= 0x3f;
  
  if (addr & 0x20) c64dtv_blitter_store(addr & 0x1f, value);
  else c64dtv_dma_store(addr, value);
}


/* ------------------------------------------------------------------------- */
static int set_dtvrevision(int val, void *param)
{
    switch (val) {
    default:
    case 3:
        dtvrevision=3;
        break;
    case 2:
        dtvrevision=2;
        break;
    }
    have_blitter_bug = (dtvrevision == 2) ? 1 : 0;
    return 1;
}

static int set_blitter_log(int val, void *param)
{
    blitter_log_enabled = val;
    return 1;
}

static const resource_int_t resources_int[] = {
    { "DtvRevision", 3, RES_EVENT_SAME, NULL,
      &dtvrevision, set_dtvrevision, NULL },
    { "DtvBlitterLog", 0, RES_EVENT_NO, (resource_value_t)0,
      &blitter_log_enabled, set_blitter_log, NULL },
    { NULL }
};

int c64dtvblitter_resources_init(void)
{
    return resources_register_int(resources_int);
}

void c64dtvblitter_resources_shutdown(void)
{
}

#ifdef HAS_TRANSLATION
static const cmdline_option_t cmdline_options[] =
{
    { "-dtvrev", SET_RESOURCE, 1, NULL, NULL, "DtvRevision", NULL,
      IDCLS_P_REVISION, IDCLS_SPECIFY_DTV_REVISION },
    { "-dtvblitterlog", SET_RESOURCE, 0, NULL, NULL, "DtvBlitterLog",
      (resource_value_t)1, 0, IDCLS_ENABLE_DTV_BLITTER_LOG },
    { "+dtvblitterlog", SET_RESOURCE, 0, NULL, NULL, "DtvBlitterLog",
      (resource_value_t)0, 0, IDCLS_DISABLE_DTV_BLITTER_LOG },
    { NULL }
};
#else
static const cmdline_option_t cmdline_options[] =
{
    { "-dtvrev", SET_RESOURCE, 1, NULL, NULL, "DtvRevision", NULL,
      N_("<revision>"), N_("Specify DTV Revision (2: DTV2, 3: DTV3)") },
    { "-dtvblitterlog", SET_RESOURCE, 0, NULL, NULL, "DtvBlitterLog",
      (resource_value_t)1, NULL, N_("Enable DTV blitter logs.") },
    { "+dtvblitterlog", SET_RESOURCE, 0, NULL, NULL, "DtvBlitterLog",
      (resource_value_t)0, NULL, N_("Disable DTV blitter logs.") },
    { NULL }
};
#endif

int c64dtvblitter_cmdline_options_init(void)
{
    return cmdline_register_options(cmdline_options);
}
