/*
 * c64dtvcpu.c - Emulation of the C64DTV processor.
 *
 * Written by
 *  M.Kiesel <mayne@users.sourceforge.net>
 *  Hannu Nuotio <hannu.nuotio@pp.inet.fi>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include "vice.h"

#include "mem.h"

/* ------------------------------------------------------------------------- */

/* MACHINE_STUFF should define/undef

 - NEED_REG_PC
 - TRACE

 The following are optional:

 - PAGE_ZERO
 - PAGE_ONE
 - STORE_IND
 - LOAD_IND
 - DMA_FUNC
 - DMA_ON_RESET

*/

/* ------------------------------------------------------------------------- */

#ifndef C64DTV
#define C64DTV
#endif

BYTE dtv_registers[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
BYTE dtvrewind;

#define REWIND_FETCH_OPCODE(clock) clock-=dtvrewind

/* Burst mode implementation */

BYTE burst_status, burst_diff, burst_idx, burst_fetch, burst_broken;
BYTE burst_cache[] = {0, 0, 0, 0};
WORD burst_addr, burst_last_addr;

inline static void c64dtvcpu_clock_add(CLOCK *clock, int amount)
{
    int count = amount; 
    while (burst_diff && count) {count--; burst_diff--;}
    *clock += count; 
}

#define CLK_ADD(clock, amount) c64dtvcpu_clock_add(&clock, amount)

/* Skip cycle implementation */

#define CLK_RTS (3 - (dtv_registers[9]&1))
#define CLK_RTI (4 - 2*(dtv_registers[9]&1))
#define CLK_BRK (5 - (dtv_registers[9]&1))
#define CLK_ABS_I_STORE2 (2 - (dtv_registers[9]&1))
#define CLK_STACK_PUSH (1 - (dtv_registers[9]&1))
#define CLK_STACK_PULL (2 - 2*(dtv_registers[9]&1))
#define CLK_ABS_RMW2 (3 - (dtv_registers[9]&1))
#define CLK_ABS_I_RMW2 (3 - (dtv_registers[9]&1))
#define CLK_ZERO_I_STORE (2 - (dtv_registers[9]&1))
#define CLK_ZERO_I2 (2 - (dtv_registers[9]&1))
#define CLK_ZERO_RMW (3 - (dtv_registers[9]&1))
#define CLK_ZERO_I_RMW (4 - 2*(dtv_registers[9]&1))
#define CLK_IND_X_RMW (3 - (dtv_registers[9]&1))
#define CLK_IND_Y_RMW1 (1 - (dtv_registers[9]&1))
#define CLK_IND_Y_RMW2 (3 - (dtv_registers[9]&1))
#define CLK_BRANCH2 (1 - (dtv_registers[9]&1))
#define CLK_INT_CYCLE (1 - (dtv_registers[9]&1))
#define CLK_JSR_INT_CYCLE (1 - (dtv_registers[9]&1) + ((dtv_registers[9]&2)&&(reg_pc==0))?1:0)
#define CLK_IND_Y_W (2 - (dtv_registers[9]&1))
#define CLK_NOOP_ZERO_X (2 - (dtv_registers[9]&1))

#define IRQ_CYCLES (7 - 2*(dtv_registers[9]&1))
#define NMI_CYCLES (7 - 2*(dtv_registers[9]&1))


/* New opcodes */

#define SAC(op) \
    do { \
        reg_a_write_idx = op >> 4; \
        reg_a_read_idx = op & 0x0f; \
        INC_PC(2); \
    } while (0)

#define SIR(op) \
    do { \
        reg_y_idx = op >> 4; \
        reg_x_idx = op & 0x0f; \
        INC_PC(2); \
    } while (0)

#define NOOP_ABS_Y()   \
  do {                 \
      LOAD_ABS_Y(p2);  \
      CLK_ADD(CLK,1);  \
      INC_PC(3);       \
  } while (0)

#define BRANCH(cond, value)                                  \
  do {                                                       \
      INC_PC(2);                                             \
                                                             \
      if (cond) {                                            \
          unsigned int dest_addr;                            \
                                                             \
          burst_broken=1;                                    \
          dest_addr = reg_pc + (signed char)(value);         \
                                                             \
          LOAD(reg_pc);                                      \
          CLK_ADD(CLK,CLK_BRANCH2);                          \
          if ((reg_pc ^ dest_addr) & 0xff00) {               \
              LOAD((reg_pc & 0xff00) | (dest_addr & 0xff));  \
              CLK_ADD(CLK,CLK_BRANCH2);                      \
          } else {                                           \
              OPCODE_DELAYS_INTERRUPT();                     \
          }                                                  \
          JUMP(dest_addr & 0xffff);                          \
      }                                                      \
  } while (0)

/* Override optimizations in maincpu.c that directly access mem_ram[] */
/* We need to channel everything through mem_read/mem_store to */
/* let the DTV segment mapper (register 12-15) do its work */

#define STORE(addr, value) \
    mem_store(addr, value)

#define LOAD(addr) \
    mem_read(addr)

/* Route zero page operations through register 10 (zeropage mapper) */

#define STORE_ZERO(addr, value) \
    mem_store((((WORD) dtv_registers[10]) << 8) + ((addr) & 0xff), value)

#define LOAD_ZERO(addr) \
    mem_read((((WORD) dtv_registers[10]) << 8) + ((addr) & 0xff))

/* Route stack operations through register 11 (stack mapper) */

#define PUSH(val) (STORE((((WORD) dtv_registers[11]) << 8) + ((reg_sp--) & 0xff), val))

#define PULL()    (LOAD((((WORD) dtv_registers[11]) << 8) + ((++reg_sp) & 0xff)))


#include "../maincpu.c"

